#include <stdio.h>
#include <mpi.h>
#include <stdlib.h>
#include <math.h>
#include <stdbool.h>
#include <time.h>
#include <omp.h>

#define total_number_of_rectangles 10000 // Number of rectangles to divide interval

double interval_start;
double interval_end;
double rectangle_width;
double step;



double CalculateIntegration(double my_interval_start, double my_interval_end , 
                            int my_number_of_rectangles ,int tid , int numThreads)
{

    double start = my_interval_start + tid * (my_interval_end - my_interval_start)/numThreads;
    double end = start + (my_interval_end - my_interval_start)/numThreads;
    double result = 0.0;

    for (double i = start; i < end; i += step)
    {
        double mid = (i + i + 1) / 2.0;
        mid /= total_number_of_rectangles;
        double y = mid * mid;
        result += y * rectangle_width;
    }
    return result;
}


/**
 * here the my_interval_start represents a start of interval and my_interval_end is the end of interval
 * that we must integrate on.
 * 
 * here we will divide our sub-interval to segments and each segment will be
 * calculated by a thread.
*/

double CalculateIntegrationOMP(double my_interval_start, double my_interval_end , 
                                int my_number_of_rectangles)
{
    double sum = 0.0;
    #pragma omp parallel
    {
        int numThreads = omp_get_num_threads();
        int tid = omp_get_thread_num();
        double localAnswer = CalculateIntegration(my_interval_start, my_interval_end,
                                                    my_number_of_rectangles,
                                                    tid, numThreads);

        #pragma omp critical
        {
            sum += localAnswer;
        }
    }

    return sum;
}



int main(int argc, char* argv[])
{
    interval_start = 0;
    interval_end = 1;
    rectangle_width = (interval_end - interval_start) / total_number_of_rectangles;
    step = rectangle_width * total_number_of_rectangles;

    int my_rank, world_size;
    double sum = 0.0;
    double localSum = 0.0;

    MPI_Init(&argc, &argv);
    MPI_Comm_rank(MPI_COMM_WORLD, &my_rank);
    MPI_Comm_size(MPI_COMM_WORLD, &world_size);

    /**
        each Machine in MPI World will have a segment to calculate , this segment is [start, end]
        now even this segment will be divided into threads to calculate through OpenMP
        but this segment will NOT be divided into the total number of rectangles
        but will be divided into its base share (total number of rectangles/size of the world)

        so here basically for each node in the MPI world we give it a sub-interval to 
        integrate on it.
    */
    double my_start = interval_start * total_number_of_rectangles + my_rank * (interval_end - interval_start) * total_number_of_rectangles / world_size;


    double my_end = my_start + (interval_end - interval_start) * total_number_of_rectangles / world_size;

    int my_number_of_rectangles = total_number_of_rectangles/world_size;

    //printf("my rank is %d and start is %lf and end is %lf\n", my_rank , my_start , my_end);

    double time = 0.0;
    MPI_Barrier(MPI_COMM_WORLD);
    time -= MPI_Wtime();


    localSum = CalculateIntegrationOMP(my_start, my_end ,my_number_of_rectangles);

    MPI_Reduce(&localSum, &sum, 1, MPI_DOUBLE, MPI_SUM, 0, MPI_COMM_WORLD);

    time += MPI_Wtime();


    if (my_rank == 0)
    {
        printf("\n\nNumerical Integral using MPI: value is %lf and time taken = %lf\n", sum , time);
        printf("*****This Program is written by Mohammed Salameh, It Works for Any Interval*** \n\n\n");
    }

    MPI_Finalize();

    return 0;
}